program percolate

  !
  !  Program to test for percolation of a cluster
  !

  use uni

  implicit none

  integer, parameter                 :: L = 20  ! System size
  integer, dimension(0:L+1, 0:L+1)   :: map     ! System state map

  integer :: loop, nchange
  logical :: percflag

  real    :: rho
  integer :: seed

  !
  ! Set the most important variable, the density rho
  !

  rho = 0.40

  !
  ! Set the randum number seed and initialise the generator
  !

  seed = 1564
  call rinit(seed)

  write(*,*) 'percolate: parameters are rho=', rho, ', L=', L, ', seed=', seed

  !
  ! Fill map with density rho
  !

  call perc_fill(L, map, rho)

  !
  ! Initialise map ready for updating
  !

  call perc_init(L, map)

  !
  ! Keep updating until nothing changes
  !

  loop = 1

  do

    call perc_update(L, map, nchange)

    write(*,*) 'percolate: number of changes on loop ', loop, ' is ', nchange

    if (nchange == 0) exit

    loop = loop + 1

  end do

  !
  ! Test to see if percolation occurred
  !

  call perc_test(L, map, percflag)

  if (percflag) then

    write(*,*) 'percolate: cluster DOES percolate'

  else

    write(*,*) 'percolate: cluster DOES NOT percolate'

  end if

  call perc_write(L, map, 'map.pgm', L*L)

end program percolate

  !
  ! Function to fill the percolation map with density rho.
  !
  ! All boundary points should be set to zero.
  !
  ! The interior n*n entries should each be set to zero with
  ! probability rho, otherwise should be set to one.
  !

  subroutine perc_fill(n, percmap, rho)

    use uni, only: random_uniform
    
    integer, intent(in)                             :: n
    integer, dimension(0:n+1, 0:n+1), intent(inout) :: percmap
    real, intent(in)                                :: rho

    integer :: i, j

    do i = 0, n+1
       do j = 0, n+1

          percmap(i,j) = mod(i+j,2)

       end do
    end do

  end subroutine perc_fill

  !
  !
  ! Function to initialise the map prior to looking for clusters.
  !
  ! All interior L*L points that are non-zero should be replaced with a
  ! unique positive integer.
  !

  subroutine perc_init(n, percmap)

    integer, intent(in)                             :: n
    integer, dimension(0:n+1, 0:n+1), intent(inout) :: percmap

  end subroutine perc_init


  !
  ! Function to update all entries of the map once.
  !
  ! Should return the actual number of changes, zero if nothing changes.
  !

  subroutine perc_update(n, map, nchange)

    integer, intent(in)                             :: n
    integer, dimension(0:n+1, 0:n+1), intent(inout) :: map
    integer, intent(out)                            :: nchange

    nchange = 0

  end subroutine perc_update


  !
  ! Function to test for percolation.
  !
  ! Should return 1 if a cluster percolates, 0 otherwise.
  !

  subroutine perc_test(n, percmap, percflag)

    integer, intent(in)                           :: n
    integer, dimension(0:n+1, 0:n+1), intent(in)  :: percmap
    logical, intent(out)                          :: percflag

    percflag = .false.

  end subroutine perc_test


  !
  ! You don't need to change the routines below
  !
  ! Routine to write percolation data to file
  !
  ! Clusters are coloured by size with the largest "maxcluster" being
  ! displayed.
  !

  subroutine perc_write(n, percmap, percfile, maxcluster)

    integer, intent(in)                          :: n
    integer, dimension(0:n+1, 0:n+1), intent(in) :: percmap
    character (len = *), intent(in)              :: percfile
    integer, intent(in)                          :: maxcluster

    integer :: lmaxcluster

    integer :: i, j
    integer :: ncluster, maxsize, colour
    integer, parameter :: fmtlen = 32

    character (len = fmtlen)  :: fmtstring

    integer, dimension(n*n)   :: rank
    integer, dimension(2,n*n) :: clustlist
    integer, dimension(n)     :: pgmline

    ! clustlist(1,:) is the size of the cluster, clustlist(2,:) is its id

    integer :: iounit = 12

    do i = 1, n*n

       rank(i) = -1

       clustlist(1,i) = 0
       clustlist(2,i) = i

    end do

    !
    ! Find the sizes of all the distinct clusters
    !

    do i = 1, n
       do j = 1, n

          if (percmap(i,j) > 0) then
             clustlist(1,percmap(i,j)) = clustlist(1,percmap(i,j)) + 1
          end if

       end do
    end do

    !
    ! Now sort them with the largest first
    !

    call perc_sort(clustlist, n*n)

    maxsize = clustlist(1,1)

    ncluster = 0

    do while (ncluster < n*n .and. clustlist(1,ncluster+1) > 0)
       ncluster = ncluster + 1
    end do

    lmaxcluster = maxcluster

    if (lmaxcluster > ncluster) then
       lmaxcluster = ncluster
    end if

    do i = 1, ncluster
       rank(clustlist(2,i)) = i
    end do

    write(*,*) 'percwrite: opening file ', percfile

    open(unit=iounit, file=percfile)

    write(*,*) 'percwrite: map has ', ncluster, &
         ' clusters, maximum cluster size is ', maxsize

    if (lmaxcluster == 1) then
       write(*,*) 'percwrite: displaying the largest cluster'
    else if (lmaxcluster == ncluster) then
       write(*,*) 'percwrite: displaying all clusters'
    else
       write(*,*) 'percwrite: displaying largest ', lmaxcluster, ' clusters'
    end if

    write(*,*) 'percwrite: writing data ...'

    write(fmtstring, fmt='(''('', i3, ''(1x, i3))'')') n
    write(iounit,fmt='(''P2'')')
    write(iounit,'(i4,i4)') n,  n

    if (lmaxcluster > 0) then
       write(iounit,*) lmaxcluster
    else
       write(iounit,*) 1
    end if

    do j = n, 1, -1
       do i = 1, n

          colour = percmap(i,j)

          !
          ! If it is part of a cluster, look for the colour
          !

          if (percmap(i,j) > 0) then

             colour = rank(percmap(i,j)) - 1

             if (colour >= lmaxcluster) then
                colour = lmaxcluster
             end if

          else
             colour = lmaxcluster
          end if

          pgmline(i) = colour

       end do

       write(iounit,fmt=fmtstring) (pgmline(i), i=1,n)

    end do

    write(*,*) 'percwrite: ... done'

    close(iounit)
    write(*,*) 'percwrite: file closed'

  end subroutine perc_write


  
  subroutine perc_sort(clustlist, n)

    integer, intent(in)                    :: n
    integer, dimension(2,n), intent(inout) :: clustlist

    integer, parameter  :: intsize = 4
    
    call sort(clustlist, 1, n, n)

  end subroutine perc_sort




  recursive subroutine sort(clustlist, begin, end, n)

    integer, intent(in) :: begin, end, n
    integer, dimension(2,n), intent(inout) :: clustlist

    integer pivot1, pivot2, left, right, pivotindex

    pivotindex = begin + (end - begin) / 2

    if(end > begin) then
       call partition(clustlist, begin, end, pivotindex, n)
       call sort(clustlist, begin, pivotindex-1, n)
       call sort(clustlist,pivotindex + 1, end, n)
    end if

  end subroutine sort


  subroutine partition(clustlist, begin, end, pivotindex, n)

    integer, intent(in) :: begin, end, n
    integer, intent(inout) :: pivotindex
    integer, dimension(2,n), intent(inout) :: clustlist
    
    integer pivot1, pivot2, left, right, i, indexpoint

    pivot1 = clustlist(1,pivotindex)
    pivot2 = clustlist(2,pivotindex)
    call swap(clustlist, end, pivotindex, n)

    left = begin
    right = end - 1
    indexpoint = left

    do i=left,right
       if(clustlist(1,i) .ge. pivot1) then
          if(clustlist(1,i) .eq. pivot1 .and. clustlist(2,i) .lt. pivot2) then
          else
             call swap(clustlist, i, indexpoint, n)
             indexpoint = indexpoint + 1
          end if
       end if
    end do

    call swap(clustlist, indexpoint, end, n)

    pivotindex = indexpoint

  end subroutine partition

  subroutine swap(clustlist, firstpoint, secondpoint, n)

    integer, intent(in) :: firstpoint, secondpoint, n
    integer, dimension(2,n), intent(inout) :: clustlist

    integer :: tempdata1, tempdata2

    tempdata1 = clustlist(1,firstpoint)
    tempdata2 = clustlist(2,firstpoint)
    clustlist(1,firstpoint) = clustlist(1,secondpoint)
    clustlist(2,firstpoint) = clustlist(2,secondpoint)
    clustlist(1,secondpoint) = tempdata1
    clustlist(2,secondpoint) = tempdata2


  end subroutine swap
